# -*- coding: utf-8 -*-
# Sentinel-2 L2A – SWIR-Visualisierung
# Fixe Bodenauflösung: 7.3 m/px
# Fixe Tile-Größe:     1086 x 729 px

from math import ceil
from pathlib import Path
import shutil
from PIL import Image
import numpy as np

try:
    import rasterio
    from rasterio.merge import merge as rio_merge
    from rasterio.transform import from_bounds
    RASTERIO_OK = True
except Exception:
    RASTERIO_OK = False

from sentinelhub import (
    SHConfig, BBox, CRS, MimeType, DataCollection, SentinelHubRequest
)

# -----------------------------
# 0) Konfiguration
# -----------------------------
config = SHConfig()
config.sh_client_id = "hier Client ID eintragen"
config.sh_client_secret = "Hier Client Secret eintragen"

# -----------------------------
# 1) Parameter eintragen, gerne für weitere Erzeugung von Testbildern nutzen
# -----------------------------
TIME_INTERVAL = ("2018-08-07", "2018-08-08") 
# BBox(bbox=(lon_min, lat_min, lon_max, lat_max)
# AOI_WGS = BBox(bbox=(8.6318, 39.8158, 9.6371, 40.3783), crs=CRS.WGS84) # SARDINIEN
# AOI_WGS = BBox(bbox=(23.3796, 38.0794, 23.8805, 38.3247), crs=CRS.WGS84) # GRIECHENLAND
# AOI_WGS = BBox(bbox=(-8.7822, 37.1231, -7.7927, 37.4770), crs=CRS.WGS84) # PORTUGAL
AOI_WGS = BBox(bbox=(12.4379, 51.5984, 14.4360, 52.3546), crs=CRS.WGS84) # BRANDENBURG
# AOI_UTM = AOI_WGS.transform(CRS.UTM_32N) # SARDINIEN
# AOI_UTM = AOI_WGS.transform(CRS.UTM_34N) # GRIECHENLAND
# AOI_UTM = AOI_WGS.transform(CRS.UTM_29N) # PORTUGAL
AOI_UTM = AOI_WGS.transform(CRS.UTM_33N) # BRANDENBURG
#___________________________________


RES_M = 7.3
TGT_W = 1086
TGT_H = 729

TILE_W_M = TGT_W * RES_M
TILE_H_M = TGT_H * RES_M

OUT = Path("downloads_final")
PNG_TREE_DIR = OUT / "png_tree"      # Sentinel Hub legt hier die Hash-Unterordner an
IMG_FLAT_DIR = OUT / "Bilder"        
TIF_DIR = OUT / "tiles_tif"          

PNG_TREE_DIR.mkdir(parents=True, exist_ok=True)
IMG_FLAT_DIR.mkdir(parents=True, exist_ok=True)
TIF_DIR.mkdir(parents=True, exist_ok=True)


DO_TIF = False       
DO_MOSAIC = False     

# -----------------------------
# 2) Evalskripte
# -----------------------------
# SWIR-Visualisierung als 8-bit RGB 
EVAL_VIS = """
//VERSION=3
function setup() {
  return {
    input: [{ bands: ["B12","B8A","B04","dataMask"], units: "REFLECTANCE" }],
    output: { bands: 3, sampleType: "UINT8" }   //  3 Bänder, 8-bit
  };
}
function evaluatePixel(s) {
  // leichte Verstärkung, hart clampen in [0,255]
  let r = Math.round(255.0 * Math.min(1, 2.5 * s.B12));
  let g = Math.round(255.0 * Math.min(1, 2.5 * s.B8A));
  let b = Math.round(255.0 * Math.min(1, 2.5 * s.B04));
  // Falls kein Datenwert: schwarz (alternativ: return [0,0,0] wenn s.dataMask==0)
  if (s.dataMask === 0) { r = 0; g = 0; b = 0; }
  return [r, g, b];
}
"""

# (optional) wissenschaftlicher Stack
EVAL_STACK = """
//VERSION=3
function setup(){
  return {
    input: [{ bands: ["B11","B12"], units: "REFLECTANCE" }],
    output: { bands: 2, sampleType: "FLOAT32" }
  };
}
function evaluatePixel(s){ return [s.B11, s.B12]; }
"""

# JPG-MimeType – kompatibel wählen (JPG oder JPEG je nach Version)
MIME_JPG = getattr(MimeType, "JPG", getattr(MimeType, "JPEG", MimeType.PNG))

# -----------------------------
# 3) Tiles planen
# -----------------------------
minx, miny, maxx, maxy = AOI_UTM
width_m  = maxx - minx
height_m = maxy - miny
cols = max(1, ceil(width_m  / TILE_W_M))
rows = max(1, ceil(height_m / TILE_H_M))

tiles = []
for r in range(rows):
    for c in range(cols):
        x0 = minx + c * TILE_W_M
        x1 = minx + (c + 1) * TILE_W_M
        y0 = miny + r * TILE_H_M
        y1 = miny + (r + 1) * TILE_H_M
        tiles.append(((r, c), BBox((x0, y0, x1, y1), crs=AOI_UTM.crs)))

# -----------------------------
# 4) Download pro Tile (fixe size)
# -----------------------------
img_paths = []
tif_paths = []

for (r, c), tile in tiles:
    tile_tree = PNG_TREE_DIR / f"r{r}_c{c}"
    tile_tree.mkdir(parents=True, exist_ok=True)

    # VIS: direkt als JPEG anfordern 
    req_jpg = SentinelHubRequest(
        evalscript=EVAL_VIS,
        input_data=[SentinelHubRequest.input_data(
            data_collection=DataCollection.SENTINEL2_L2A,
            time_interval=TIME_INTERVAL,
            mosaicking_order="leastCC",
            upsampling="BILINEAR"
        )],
        responses=[SentinelHubRequest.output_response("default", MIME_JPG)],
        bbox=tile,
        size=(TGT_W, TGT_H),
        data_folder=str(tile_tree),
        config=config
    )
    _ = req_jpg.get_data(save_data=True)

    # response.jpg finden
    response_candidates = sorted(
        list(tile_tree.rglob("response.jpg")) +
        list(tile_tree.rglob("response.jpeg")) +
        list(tile_tree.rglob("response.png"))
    )
    if not response_candidates:
        raise RuntimeError(f"Keine response-Datei für Tile r{r}_c{c}")

    response_src = response_candidates[-1]
    # Wenn PNG zurückkam, wandeln wir on-the-fly in JPG um
    flat_name = f"tile_r{r}_c{c}.jpg"
    dst = IMG_FLAT_DIR / flat_name
    if response_src.suffix.lower() == ".png":
        im = Image.open(response_src).convert("RGB")
        im.save(dst, format="JPEG", quality=85, optimize=True)  # 🔹 ~150–300 KB
    else:
        # ggf. nochmals re-encoden, um kleine Größe zu erzwingen
        im = Image.open(response_src).convert("RGB")
        im.save(dst, format="JPEG", quality=85, optimize=True)
    img_paths.append((r, c, str(dst)))

    # optionaler TIFF-Stack (für Klassifikation i.d.R. nicht nötig)
    if DO_TIF:
        req_tif = SentinelHubRequest(
            evalscript=EVAL_STACK,
            input_data=[SentinelHubRequest.input_data(
                data_collection=DataCollection.SENTINEL2_L2A,
                time_interval=TIME_INTERVAL,
                mosaicking_order="leastCC",
                upsampling="BILINEAR"
            )],
            responses=[SentinelHubRequest.output_response("default", MimeType.TIFF)],
            bbox=tile,
            size=(TGT_W, TGT_H),
            data_folder=str(TIF_DIR / f"r{r}_c{c}"),
            config=config
        )
        arr = req_tif.get_data(save_data=True)[0]
        if RASTERIO_OK:
            tif_file = TIF_DIR / f"tile_r{r}_c{c}.tif"
            with rasterio.open(
                tif_file, "w",
                driver="GTiff",
                height=TGT_H,
                width=TGT_W,
                count=2,
                dtype=arr.dtype,
                crs=tile.crs.pyproj_crs(),
                transform=from_bounds(*tile, TGT_W, TGT_H)
            ) as dst:
                dst.write(arr[:, :, 0], 1)
                dst.write(arr[:, :, 1], 2)
            tif_paths.append(str(tif_file))

# -----------------------------
# 5) (Optional) Mosaik aus JPGs
# -----------------------------
if DO_MOSAIC and img_paths:
    rows_set = rows
    cols_set = cols
    row_images = []
    for r in range(rows_set):
        row_entries = sorted([(rr, cc, pth) for (rr, cc, pth) in img_paths if rr == r], key=lambda x: x[1])
        imgs = [Image.open(pth).convert("RGB") for _, _, pth in row_entries]
        tw, th = imgs[0].size
        row_canvas = Image.new("RGB", (tw * len(imgs), th))
        x = 0
        for im in imgs:
            row_canvas.paste(im, (x, 0))
            x += tw
        row_images.append(row_canvas)

    tw, th = row_images[0].size[0] // cols_set, row_images[0].size[1]
    mosaic_png = Image.new("RGB", (tw * cols_set, th * rows_set))
    y = 0
    for im in row_images:
        mosaic_png.paste(im, (0, y))
        y += th
    mosaic_path = OUT / "mosaic_vis.jpg" 
    mosaic_png.save(mosaic_path, format="JPEG", quality=85, optimize=True)
    print(f"✅ JPG-Mosaik: {mosaic_path}")

# -----------------------------
print("✅ Kacheln (JPG):", IMG_FLAT_DIR)
if DO_TIF and tif_paths:
    print("✅ TIFF-Tiles:", TIF_DIR)
